/*
 * HC_SR04Kernel.c
 *
 *  Created on: May 1, 2015
 *      Author: jcmx9
 */

#ifndef MODULE
#define MODULE
#endif

#ifndef __KERNEL__
#define __KERNEL__
#endif

#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <rtai.h>
#include <rtai_sched.h>
#include <rtai_hal.h>
#include <rtai_fifos.h>

MODULE_LICENSE("GPL");

static RT_TASK trigger;
RTIME period;

//use hardware interrupt Line 59, refer to GPIOINTR, the combined interrupt from any bit in ports A or B
int hrd_irq_num = 59;

//registers pointers
unsigned long *PADR;									//Port A Data Register
unsigned long *PADDR;									//Port A Data Direction Register
unsigned long *GPIOAIntEn;								//Interrupt Enable Register A
unsigned long *GPIOAIntType1;							//Interrupt Type 1 Register A
unsigned long *GPIOAIntType2;							//Interrupt Type 2 Register A
unsigned long *GPIOAIntEOI;								//End-Of_Interrupt Register A
unsigned long *GPIOADB;									//debouncing register of PortA
unsigned long *IntStsA;									//Interrupt Status Register

//interrupt handler
static void hwdHandler(unsigned irq_num, void *cookie){

	struct timeval tv;									//timeval variable to record time
	int rising = 1;										//rising edge flag for fifo1
	int falling = 0;									//falling edge flag for fifo1
	rt_disable_irq (irq_num);							//disable interrupt when handler is executed

	do_gettimeofday(&tv);								//get timestamp

	if ( (*GPIOAIntType2 >> 1) & 1)						//if the interrupt caused by rising edge
		rtf_put (1, &rising, sizeof (int));				//put a rising edge flag, 1, in the fifo1
	else
		rtf_put (1, &falling, sizeof (int));			//else, put a falling edge flag, 0, in the fifo1

	rtf_put (0, &tv, sizeof(struct timeval));			//put timestamp in fifo0

	*GPIOAIntType2 ^= (1 << 1);		//toggle so it will switch between rising and falling edge sensitive
	*GPIOAIntEOI |= 1 << 1;			//clear hardware interrupt on portB1

	rt_enable_irq (irq_num);		//enable interrupt after handler has been executed
}

static void rt_process(int t){
	while (1){
		*PADR |= 1 << 0;				//set HIGH to trigger signal
		rt_sleep (period);				//trigger pulse should be at lease 10us, 2ms in this case
		*PADR &= ~(1 << 0);				//set low to end the pulse

		rt_task_wait_period ();			//wait for predefined real-time period, 10ms in this case
	}
}

int init_module(void){
	//map address
	unsigned long *GPOI_base;		//GPIO Control Registers
	GPOI_base = (unsigned long *)__ioremap(0x80840000, 4096, 0);	//map address to virtual memory

	//char is 1 byte long, this conversion make me easy to calculate the offset from datasheet value
	PADR = (unsigned long *)((char *)GPOI_base + 0x00);
	PADDR = (unsigned long *)((char *)GPOI_base + 0x10);
	GPIOAIntEn = (unsigned long *)((char *)GPOI_base + 0x9C);
	GPIOAIntType1 = (unsigned long *)((char *)GPOI_base + 0x90);
	GPIOAIntType2 = (unsigned long *)((char *)GPOI_base + 0x94);
	GPIOAIntEOI = (unsigned long *)((char *)GPOI_base + 0x98);
	GPIOADB = (unsigned long *)((char *)GPOI_base + 0xA8);
	IntStsA = (unsigned long *)((char *)GPOI_base + 0xA0);

	//modify registers
	*PADDR &= ~(1 << 1);				//set bit 1 to input for receiving "Echo" pulse
	*PADDR |= 1 << 0;					//set bit 0 to output for "Trig" pulse

	*PADR &= ~(1 << 0);					//clear 0 bit to make sure no incorrect pulse before trigger

	*GPIOAIntType1 |= 1 << 1;			//initiate portA 1 (echo) as edge sensitive
	*GPIOAIntType2 |= 1 << 1;			//initiate portA 1 (echo) as rising edge sensitive
	*GPIOADB |= 1 << 1;					//enable portA 1 (echo) debouncing

	rtf_create(0, sizeof(struct timeval));		//create a timeval size fifo0 for timestamp
	rtf_create (1, sizeof (int));				//create a int size fifo1 for edge flag

	//initiate real-time task
	rt_set_periodic_mode();							//set the real-time process to periodic mode
	period = start_rt_timer(nano2count(2000000));	//internal period is 2ms, minimum counter 100000
	rt_task_init(&trigger, rt_process, 0, 512, 0, 0, 0);			//bind the real-time task with handler
	rt_task_make_periodic (&trigger, rt_get_time(), 5 * period);	//distance measuring internal is 10ms

	//initiate interrupt
	rt_request_irq (hrd_irq_num, hwdHandler, 0, 1);			//request irq line for the
	*GPIOAIntEOI |= 1 << 1;									//clear portA1 interrupt
	*GPIOAIntEn = 0x02;										//enable interrupt on portA1
	rt_enable_irq (hrd_irq_num);							//enable interrupt on this irq line

	printk ("HC_SR04Kernel MODULE INSTALLED\n");
	return 0;
}

void cleanup_module(void){
	*GPIOAIntEOI |= 1 << 1;									//clear interrupt on bit 1 of Port A
	rt_disable_irq (hrd_irq_num);							//disable interrupt on this irq line
	rt_release_irq (hrd_irq_num);							//release the requested irq line
	rtf_destroy(0);											//destroy fifo0
	rtf_destroy(1);											//destroy fifo1
	rt_task_delete(&trigger);								//delete the realtime process
	stop_rt_timer();										//stop timer for realtime process
	printk ("HC_SR04Kernel MODULE REMOVED\n");
}
